/*
	gfx.c

	This is a graphics engine capable of a couple different
	rendering styles. It can also swap out on the fly, but
	could be very buggy, I dunno.

	Perhaps of all the modules, this is the most over-zealous.
	Considering there is only need for one "gfx_engine", the
	create, destroy, etc functions are abit much. But I was
	in the module mood.
*/

#include <allegro.h>

//#include "main.h"

#include "gfx.h"

char *gfx_name[] = {"NONE", "DOUBLE BUFFER", "PAGE FLIP"};
int stretch = 0;
int stretch_x, stretch_y, stretch_w, stretch_h;

/*
	Creates the graphics engine. You must pass the type
	of updating you would like.

	This could use a lot of error handling!
*/
GFX *gfx_create(int type)
{
	return gfx_create_ex(type, SCREEN_W, SCREEN_H, get_color_depth());
}

/*
	Creates the graphics engine. You must pass the type
	of updating you would like.

	This could use a lot of error handling!
*/
GFX *gfx_create_ex(int type, int width, int height, int color_depth)
{
	GFX *gfx = malloc(sizeof(GFX));
	int error = 0;

	(void)color_depth;

	gfx->type = type;
	gfx->overlays = NULL;
	
	switch (type)
	{
		case GFX_DOUBLE_BUFFER:
			gfx->pages = malloc(sizeof(BITMAP*));
			gfx->num_pages = 1;
			gfx->pages[0] = create_bitmap(width, height);
			gfx->buffer = gfx->pages[0];
			gfx->current_page = 0;
			
			if (!gfx->pages[0]) error = 1;
		break;

		case GFX_PAGE_FLIP:
			gfx->pages = malloc(sizeof(BITMAP*)*2);
			gfx->num_pages = 2;
			gfx->pages[0] = create_video_bitmap(SCREEN_W, SCREEN_H);
			gfx->pages[1] = create_video_bitmap(SCREEN_W, SCREEN_H);
			gfx->buffer = gfx->pages[0];
			gfx->current_page = 0;

			if (!gfx->pages[0] || !gfx->pages[1]) error = 1;
		break;

		default:
			gfx->num_pages = 0;
			gfx->pages = 0;
		break;
	}

	if (error)
	{
		gfx_destroy(gfx);
		gfx = 0;
	}

	if (width != SCREEN_W || height != SCREEN_H)
	{
		stretch = 1;
		if (SCREEN_W / (float)width < SCREEN_H / (float)height)
		{
			stretch_w = SCREEN_W;
			stretch_h = ((float)SCREEN_W / (float)width) * height;
		}
		else
		{
			stretch_h = SCREEN_H;
			stretch_w = ((float)SCREEN_H / (float)height) * width;			
		}
		stretch_x = (SCREEN_W - stretch_w) / 2;
		stretch_y = (SCREEN_H - stretch_h) / 2;
	}

	
	return gfx;
}


/*
	Destroys the engine and frees all memory from the bitmaps
*/
void gfx_destroy(GFX *gfx)
{
	if (gfx)
	{
		if (gfx->pages)
		{
			while (--gfx->num_pages >= 0)			
				if (gfx->pages[gfx->num_pages]) destroy_bitmap(gfx->pages[gfx->num_pages]);			
			free(gfx->pages);
			gfx->pages = 0;
		}

		while (gfx->overlays)
		{
			GFX_OVERLAY *tmp = gfx->overlays;
			free(gfx->overlays);
			gfx->overlays = tmp;
		}

		free(gfx);		
	}
}

/*
	This needs to be called before you draw any graphics to aa
	new frame.
*/
BITMAP *gfx_frame_start (GFX *gfx)
{
	switch (gfx->type)
	{
		case GFX_DOUBLE_BUFFER:
			clear(gfx->buffer);
		break;

		case GFX_PAGE_FLIP:
			clear(gfx->buffer = gfx->pages[gfx->current_page = 1 - gfx->current_page]);
		break;
	}

	return gfx->buffer;
}

/*
	This draws the graphics to the screen.
*/
void gfx_frame_end (GFX *gfx)
{
	GFX_OVERLAY *overlay = gfx->overlays;
	while (overlay)
	{
		if (overlay->draw_mode == 1)
		{
			blit(overlay->bmp, gfx->buffer, 0,0, overlay->x, overlay->y, overlay->bmp->w, overlay->bmp->h);
		}
		overlay = overlay->next;
	}

	switch (gfx->type)
	{
		case GFX_DOUBLE_BUFFER:
			acquire_screen();
			if (!stretch)
				blit(gfx->buffer, screen, 0,0, 0,0, SCREEN_W, SCREEN_H);
			else
				stretch_blit(gfx->buffer, screen, 0,0, gfx->buffer->w,gfx->buffer->h, stretch_x,stretch_y,stretch_w,stretch_h);
			release_screen();			
		break;

		case GFX_PAGE_FLIP:
			show_video_bitmap(gfx->buffer);			
		break;
	}
}

/*
	This is a useful function, but is very ugly, and perhaps buggy.
	It lets you switch between different types of updating on the fly.
*/
void gfx_switch_type (GFX **gfx, int type)
{
	BITMAP *bmp_temp;
	switch ((*gfx)->type)
	{
		case GFX_DOUBLE_BUFFER:			
			bmp_temp = create_bitmap(SCREEN_W, SCREEN_H);
			blit((*gfx)->buffer, bmp_temp, 0,0, 0,0, SCREEN_W, SCREEN_H);
			gfx_destroy(*gfx);

			if (!(*gfx = gfx_create(type)))
			{
				allegro_message("Unable to swap!");
				exit(1);
			}
			gfx_frame_start(*gfx);
			blit(bmp_temp, (*gfx)->buffer, 0,0 ,0,0, SCREEN_W, SCREEN_H);
			gfx_frame_end(*gfx);
			destroy_bitmap(bmp_temp);
		break;

		case GFX_PAGE_FLIP:
			bmp_temp = create_bitmap(SCREEN_W, SCREEN_H);
			blit((*gfx)->buffer, bmp_temp, 0,0, 0,0, SCREEN_W, SCREEN_H);
			gfx_destroy(*gfx);
			show_video_bitmap(screen);

			if (!(*gfx = gfx_create(type)))
			{
				allegro_message("Unable to swap!");
				exit(1);
			}
			gfx_frame_start(*gfx);
			blit(bmp_temp, (*gfx)->buffer, 0,0 ,0,0, SCREEN_W, SCREEN_H);
			gfx_frame_end(*gfx);
			destroy_bitmap(bmp_temp);
		break;
	}

	return;
}

GFX_OVERLAY *gfx_add_overlay(GFX *gfx, BITMAP *bmp, int x, int y, int draw_mode)
{
	GFX_OVERLAY *overlay = (GFX_OVERLAY *)malloc(sizeof(GFX_OVERLAY));
	overlay->bmp = bmp;
	overlay->x = x;
	overlay->y = y;
	overlay->draw_mode = draw_mode;

	overlay->next = gfx->overlays;
	gfx->overlays = overlay;
	
	return overlay;
}

void gfx_remove_overlay(GFX *gfx, GFX_OVERLAY *del_overlay)
{
	GFX_OVERLAY *overlay = gfx->overlays;
	GFX_OVERLAY *prev = NULL;

	while (overlay)
	{
		if (overlay == del_overlay)
		{
			if (prev)
				prev->next = overlay->next;
			else
				gfx->overlays = overlay->next;

			free(overlay);
			break;
		}

		overlay = overlay->next;
	}
}